#!/usr/bin/env python3
"""
Скрипт для создания инсталятора приложения "Моделирование попаданий импульса"
Использует PyInstaller для создания исполняемого файла
"""

import os
import sys
import subprocess
import shutil
from pathlib import Path

def check_pyinstaller():
    """Проверяем, установлен ли PyInstaller"""
    try:
        import PyInstaller
        return True
    except ImportError:
        return False

def install_pyinstaller():
    """Устанавливаем PyInstaller"""
    print("Устанавливаем PyInstaller...")
    subprocess.run([sys.executable, "-m", "pip", "install", "pyinstaller"], check=True)
    print("PyInstaller установлен!")

def create_executable():
    """Создаем исполняемый файл"""
    print("Создаем исполняемый файл...")
    
    # Команда для PyInstaller
    cmd = [
        "pyinstaller",
        "--onefile",                    # Один исполняемый файл
        "--windowed",                   # Без консольного окна (уберем это для интерактивного приложения)
        "--name=ImpulseSimulator",      # Имя исполняемого файла
        "--icon=icon.ico",              # Иконка (если есть)
        "--distpath=dist",              # Папка для результата
        "--workpath=build",             # Рабочая папка
        "--specpath=.",                 # Папка для .spec файла
        "app.py"
    ]
    
    # Убираем --windowed для консольного приложения
    cmd.remove("--windowed")
    
    # Убираем иконку если её нет
    if not os.path.exists("icon.ico"):
        cmd.remove("--icon=icon.ico")
    
    try:
        subprocess.run(cmd, check=True)
        print("✅ Исполняемый файл создан успешно!")
        print(f"📁 Файл находится в: {os.path.abspath('dist/ImpulseSimulator.exe')}")
    except subprocess.CalledProcessError as e:
        print(f"❌ Ошибка при создании исполняемого файла: {e}")
        return False
    
    return True

def create_portable_package():
    """Создаем портабл пакет"""
    print("Создаем портабл пакет...")
    
    portable_dir = Path("portable")
    if portable_dir.exists():
        shutil.rmtree(portable_dir)
    
    portable_dir.mkdir(exist_ok=True)
    
    # Копируем исполняемый файл
    exe_path = Path("dist/ImpulseSimulator.exe")
    if exe_path.exists():
        shutil.copy2(exe_path, portable_dir)
    
    # Создаем README для портабл версии
    portable_readme = portable_dir / "README.txt"
    with open(portable_readme, "w", encoding="utf-8") as f:
        f.write("""=== Портабл версия: Моделирование попаданий импульса ===

ИНСТРУКЦИЯ ПО ИСПОЛЬЗОВАНИЮ:

1. Запустите ImpulseSimulator.exe
2. Введите времена появления стробов (например: 0, 40, 80, 160, 240)
3. Нажмите Enter на пустой строке для завершения ввода
4. Программа покажет график и сохранит его как result.png

СИСТЕМНЫЕ ТРЕБОВАНИЯ:
- Windows 7/8/10/11
- Не требует установки Python или дополнительных библиотек

ПРИМЕЧАНИЕ:
- Результаты сохраняются в папку с программой
- График автоматически отображается после расчетов
""")
    
    print(f"✅ Портабл пакет создан в папке: {portable_dir.absolute()}")
    return True

def main():
    print("=== Сборка инсталятора для Моделирования попаданий импульса ===\n")
    
    # Проверяем PyInstaller
    if not check_pyinstaller():
        try:
            install_pyinstaller()
        except subprocess.CalledProcessError:
            print("❌ Не удалось установить PyInstaller")
            return
    
    # Создаем исполняемый файл
    if create_executable():
        # Создаем портабл пакет
        create_portable_package()
        
        print("\n=== ГОТОВО! ===")
        print("📦 Инсталятор: dist/ImpulseSimulator.exe")
        print("💾 Портабл: portable/ImpulseSimulator.exe")
        print("\nДля создания MSI инсталятора используйте create_msi_installer.py")

if __name__ == "__main__":
    main() 