import numpy as np
from collections import Counter
import sys
import os

# Настройка matplotlib для работы без GUI
import matplotlib
matplotlib.use('Agg')  # Использовать backend без GUI
import matplotlib.pyplot as plt

def simulate_impulse_hits(impulse_length=128, stroby_times=None, num_samples=100_000):
    if stroby_times is None:
        raise ValueError("stroby_times должен быть задан!")
    stroby_times = sorted(stroby_times)
    period_length = stroby_times[-1] if len(stroby_times) > 0 else 0
    # Генерируем позиции стробов на отрезке в два периода
    stroby_positions = np.concatenate([
        np.array(stroby_times),
        np.array(stroby_times) + period_length
    ])

    # Случайные старты импульса в пределах одного периода
    np.random.seed(42)
    impulse_starts = np.random.uniform(0, period_length, num_samples)

    # Подсчет попаданий
    def count_hits(t_start):
        return sum(t_start <= s < t_start + impulse_length for s in stroby_positions)

    hit_counts = [count_hits(t) for t in impulse_starts]

    # Группируем: 0, 1, 2, 3+
    summary = Counter()
    for hits in hit_counts:
        if hits == 0:
            summary['0'] += 1
        elif hits == 1:
            summary['1'] += 1
        elif hits == 2:
            summary['2'] += 1
        else:
            summary['3+'] += 1
    total = sum(summary.values())
    probabilities = {k: v / total for k, v in summary.items()}

    # Визуализация
    try:
        plt.figure(figsize=(10, 6))
        plt.bar(probabilities.keys(), probabilities.values(), color='skyblue')
        plt.title(f"Вероятности попаданий (импульс {impulse_length}, период {period_length})")
        plt.ylabel("Вероятность")
        plt.grid(True, axis='y')
        plt.xticks(rotation=0)
        plt.tight_layout()
        
        # Сохраняем график в файл
        result_file = "result.png"
        plt.savefig(result_file, dpi=150, bbox_inches='tight')
        plt.close()  # Закрываем фигуру для освобождения памяти
        
        print(f"📊 График сохранен как {result_file}")
        
        # Проверяем, создался ли файл
        if os.path.exists(result_file):
            file_size = os.path.getsize(result_file)
            print(f"📁 Размер файла: {file_size} байт")
        
    except Exception as e:
        print(f"⚠️ Предупреждение: не удалось создать график - {e}")
        print("Но расчеты выполнены успешно!")

    return probabilities

def input_stroby_times():
    print("Введите времена появления стробов в периоде (по возрастанию, через Enter).")
    print("Для завершения введите пустую строку.")
    times = []
    
    while True:
        try:
            if times:
                prompt = f"Время {len(times)+1}-го строба: "
            else:
                prompt = "Время 1-го строба: "
            
            # Более надежный ввод
            try:
                s = input(prompt).strip()
            except (EOFError, KeyboardInterrupt):
                print("\nВвод прерван пользователем.")
                if len(times) == 0:
                    return None
                break
            
            if not s:
                break
                
            try:
                t = float(s)
                times.append(t)
                print(f"✓ Добавлено время: {t}")
            except ValueError:
                print("❌ Ошибка: введите число!")
                
        except Exception as e:
            print(f"❌ Ошибка ввода: {e}")
            break
    
    if len(times) < 1:
        print("❌ Не введено ни одного строба!")
        return None
    
    return times

def main():
    print("=" * 60)
    print("   МОДЕЛИРОВАНИЕ ПОПАДАНИЙ ИМПУЛЬСА")
    print("=" * 60)
    print()
    
    # Проверяем аргументы командной строки для тестового режима
    if len(sys.argv) > 1 and sys.argv[1] == "--test":
        print("🧪 ТЕСТОВЫЙ РЕЖИМ")
        times = [0, 40, 80, 160, 240]
        print(f"Используются тестовые данные: {times}")
    else:
        times = input_stroby_times()
    
    if times:
        print(f"\n📋 Введенные времена стробов: {times}")
        print("🔄 Запуск симуляции...")
        
        try:
            probs = simulate_impulse_hits(impulse_length=128, stroby_times=times)
            
            print("\n" + "=" * 40)
            print("         РЕЗУЛЬТАТЫ")
            print("=" * 40)
            
            for k, v in probs.items():
                print(f"{k} попаданий: {v:.4f} ({v*100:.2f}%)")
            
            print("\n✅ Симуляция завершена успешно!")
            
        except Exception as e:
            print(f"\n❌ Ошибка при выполнении симуляции: {e}")
            import traceback
            traceback.print_exc()
    else:
        print("❌ Программа завершена без выполнения расчетов.")
    
    print("\nДля выхода нажмите Enter...")
    try:
        input()
    except:
        pass

if __name__ == "__main__":
    main() 